<?php

namespace FrontBundle\Controller;

use AppBundle\Entity\CoinsHistory;
use AppBundle\Entity\DeliveryAddress;
use AppBundle\Entity\Order;
use AppBundle\Entity\OrderDetails;
use AppBundle\Entity\PrintRequest;
use AppBundle\Entity\Product;
use AppBundle\Entity\QrCodeRequest;
use AppBundle\Entity\RefOrderStatus;
use AppBundle\Entity\RefPrintStatus;
use AppBundle\Entity\Subscription;
use AppBundle\Service\AccessManager;
use AppBundle\Service\MailManager;
use AppBundle\Service\QrCodeGenerator;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

class OrderLargeController extends Controller
{

    private function redirectByStatus(Order $order)
    {
        $status = $order->getStatus()->getId();
        if($status == 1)
        {
            return $this->redirectToRoute('order_choose_large', ['order_id' => $order->getId()]);
        }
        elseif($status == 2)
        {
            return $this->redirectToRoute('order_pay', ['order_id' => $order->getId()]);
        }
        elseif(in_array($status, [3]))
        {
            return $this->redirectToRoute('order_summary', ['order_id' => $order->getId()]);
        }
        else
        {
            return $this->redirectToRoute('dashboard_index');
        }

        return false;
    }

    /**
     * @Route("/dashboard/order/choose_large/{order_id}", name="order_choose_large")
     */
    public function orderChooseLargeAction(Request $request, $order_id, AccessManager $accessManager)
    {
        $order = $this->getDoctrine()->getRepository(Order::class)->findOneBy([
            'id' => $order_id
        ]);

        if(!$this->getUser()->getCompany()->getRole() == 'large'){
            return $this->redirectToRoute('dashboard_index');
        }

        if(!$accessManager->checkUserbyOrder($this->getUser(), $order))
        {
            throw new AccessDeniedHttpException('Vous ne pouvez pas modifier cette commande.');
        }

        if($order->getStatus()->getId() != 1)
        {
            return $this->redirectByStatus($order);
        }

        $products = $this->getDoctrine()->getRepository(Product::class)->findAll();

        return $this->render('Front/Dashboard/order/choose_large.html.twig', [
            'order' => $order,
            'products' => $products
        ]);
    }

    /**
     * @Route("/dashboard/order/confirm_large/{order_id}", name="order_confirm_large")
     */
    public function orderConfirmLargeAction(Request $request, $order_id, AccessManager $accessManager, MailManager $mailManager)
    {
        $order = $this->getDoctrine()->getRepository(Order::class)->findOneBy([
            'id' => $order_id
        ]);

        if(!$this->getUser()->getCompany()->getRole() == 'large'){
            return $this->redirectToRoute('dashboard_index');
        }

        if (!$accessManager->checkUserbyOrder($this->getUser(), $order)) {
            throw new AccessDeniedHttpException('Vous ne pouvez pas modifier cette commande.');
        }

        $em = $this->getDoctrine()->getManager();

        $products = $request->request->get('products');
        $delivery = $request->request->get('delivery');

        if($delivery == 'custom'){
            $address = $request->request->get('delivery_address');
            if(!$address['address_1'] || !$address['postal_code'] || !$address['city']){
                $this->addFlash('warning', 'L\'adresse fournie n\'est pas complète.');
                return $this->redirectToRoute('order_choose_large', ['order_id' => $order->getId()]);
            }
        }


        if ($products) {

            # Get total price
            $total = 0;
            foreach($products as $product)
            {
                $product_object = $this->getDoctrine()->getRepository(Product::class)->findOneBy(['id' => $product]);
                $total = $total + $product_object->getPriceHt();
            }
            if($total == 0){
                $this->addFlash('warning', "Il y a une erreur dans votre commande, merci de rééssayer.");
                return $this->redirectToRoute('order_choose_large', ['order_id' => $order_id]);
            }

            # If user has enough money
            $money = $this->getUser()->getCompany()->getCoins();
            if ($total > $money) {
                $this->addFlash('warning', "Vous ne possédez pas assez de crédits pour faire cette commande.");
                return $this->redirectToRoute('order_choose_large', ['order_id' => $order_id]);
            }


            foreach($products as $product)
            {
                $product_object = $this->getDoctrine()->getRepository(Product::class)->findOneBy(['id' => $product]);

                $detail = new OrderDetails();
                $detail->setOrder($order)
                    ->setQuantity(1)
                    ->setProduct($product_object);
                $em->persist($detail);

                # If product is Registre en ligne, create subscription
                    if ($product_object->getSlug() == 'gc-registre-en-ligne') {
                    foreach ($order->getAccessibilityRegisters() as $register) {
                        $dateEnd = new \DateTime();
                        $dateEnd->modify('+1 year');

                        # Attribute QrCode to order
                        $qrCode = $this->getDoctrine()->getRepository(\AppBundle\Entity\QrCode::class)->findByAvailableQrCode();
                        if($qrCode) {
                            $em = $this->getDoctrine()->getManager();
                            $register->setQrCode($qrCode);
                            $em->persist($register);

                            $qrCode->setAssignedAt(new \DateTime());

                            $em->persist($qrCode);
                            $em->flush();
                        }

                        $qrCodeRequest = new QrCodeRequest();
                        $qrCodeRequest->setOrder($order)
                            ->setRegister($register)
                            ->setStatus('pending');
                        $em->persist($qrCodeRequest);

                        $subscription = new Subscription();
                        $subscription->setOrder($order)
                            ->setUser($this->getUser())
                            ->setErp($register->getErp())
                            ->setDateStart(new \DateTime())
                            ->setDateEnd($dateEnd);
                        $em->persist($subscription);
                        break;
                    }
                }

                if($product_object->getSlug() == 'gc-version-pdf'){
                    foreach ($order->getAccessibilityRegisters() as $register) {
                        $register->setAllowPdf(1);
                    }
                }

                if ($delivery) {
                    $order->setDeliveryChoice($delivery);
                    if($delivery == 'custom'){
                        $deliveryAddress = new DeliveryAddress();
                        $deliveryAddress->setAddress1($address['address_1'])
                            ->setAddress2($address['address_2'])
                            ->setPostalCode($address['postal_code'])
                            ->setCity($address['city']);

                        $em = $this->getDoctrine()->getManager();
                        $order->setDeliveryAddress($deliveryAddress);
                        $em->persist($deliveryAddress);
                    }
                }

                # If product is version papier, make print request
                if ($product_object->getSlug() == 'gc-version-papier') {
                    if ($delivery) {
                        if (!in_array($delivery, ['erp', 'company', 'custom'])) {
                            $this->addFlash('warning', "Il y a une erreur dans votre commande, merci de rééssayer.");
                            return $this->redirectToRoute('order_choose_large', ['order_id' => $order_id]);
                        }
                        foreach ($order->getAccessibilityRegisters() as $register) {
                            $printStatus = $this->getDoctrine()->getRepository(RefPrintStatus::class)->findOneBy(['id' => 1]);
                            $printRequest = new PrintRequest();
                            $printRequest->setOrder($order)
                                ->setRegister($register)
                                ->setRefPrintStatus($printStatus);
                            $em->persist($printRequest);
                        }
                    } else {
                        $this->addFlash('warning', "Il y a une erreur dans votre commande, merci de rééssayer.");
                        return $this->redirectToRoute('order_choose_large', ['order_id' => $order_id]);
                    }
                }


            }

            $order->setUser($this->getUser());

            foreach ($order->getAccessibilityRegisters() as $register) {
                $register->setValidatedAt(new \DateTime());
                $em->persist($register);
            }

            $company = $this->getUser()->getCompany();


            $coinsHistory = new CoinsHistory();
            $coinsHistory->setUser($this->getUser())
                ->setCompany($company)
                ->setMovementType('order')
                ->setOrder($order)
                ->setOperation('-'.$total)
                ->setOldBalance($company->getCoins())
                ->setNewBalance($company->getCoins() - $total);

            $order->setAmountTtc($total);

            $company->setCoins($company->getCoins() - $total);
            $em->persist($coinsHistory);
            $em->persist($company);

            $orderStatus = $this->getDoctrine()->getRepository(RefOrderStatus::class)->findOneBy(['id' => 3]);
            $order->setStatus($orderStatus)
                ->setValidatedAt(new \DateTime());
            $em->persist($order);

            $em->flush();

            $mailManager->sendOrderValidatedNotificationEmail($order);

            $this->addFlash('success', "Votre commande bien été passée, votre montant de crédits a été mis à jour.");
            return $this->redirectToRoute('order_summary', [
                'order_id' => $order->getId(),
            ]);
        }

        return $this->redirectToRoute('dashboard_index');

    }

}
