<?php

namespace FrontBundle\Controller;

use AppBundle\Entity\AccessibilityRegister;
use AppBundle\Entity\BonusParrainage;
use AppBundle\Entity\DeliveryAddress;
use AppBundle\Entity\Erp;
use AppBundle\Entity\FileUpload;
use AppBundle\Entity\Order;
use AppBundle\Entity\OrderDetails;
use AppBundle\Entity\Post;
use AppBundle\Entity\PrintRequest;
use AppBundle\Entity\Product;
use AppBundle\Entity\QrCodeRequest;
use AppBundle\Entity\RefOrderStatus;
use AppBundle\Entity\RefPrintStatus;
use AppBundle\Entity\Subscription;
use AppBundle\Service\AccessManager;
use AppBundle\Service\InvoiceGenerator;
use AppBundle\Service\MailManager;
use AppBundle\Service\OrderHistoryManager;
use AppBundle\Service\PdfGenerator;
use AppBundle\Service\QrCodeGenerator;
use AppBundle\Service\SubscriptionManager;
use Endroid\QrCode\QrCode;
use AppBundle\Service\OrderRedirectManager;
use FrontBundle\Form\DeliveryAddressType;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Symfony\Component\Config\Definition\Exception\Exception;
use Symfony\Component\Form\Extension\Core\Type\FormType;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

class OrderController extends Controller
{

    private function redirectByStatus(Order $order)
    {
        $status = $order->getStatus()->getId();
        if($status == 1)
        {
            return $this->redirectToRoute('order_choose', ['order_id' => $order->getId()]);
        }
        elseif($status == 2)
        {
            return $this->redirectToRoute('order_pay', ['order_id' => $order->getId()]);
        }
        elseif(in_array($status, [3]))
        {
            return $this->redirectToRoute('order_summary', ['order_id' => $order->getId()]);
        }
        else
        {
            return $this->redirectToRoute('dashboard_index');
        }

        return false;
    }

    /**
     * @Route("/dashboard/order/create/{accessibility_register_id}", name="order_create")
     */
    public function orderCreateAction(Request $request, $accessibility_register_id, AccessManager $accessManager, OrderHistoryManager $orderHistoryManager)
    {

        $accessibilityRegister = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findOneBy(
            ['id' => $accessibility_register_id]
        );

        if(!$accessManager->checkUserbyRegister($this->getUser(), $accessibilityRegister))
        {
            throw new AccessDeniedHttpException('Vous ne pouvez pas créer une commande pour ce registre.');
        }

        # If order already exist for this accessibility register
        if($accessibilityRegister->getOrder())
        {
            if(in_array($accessibilityRegister->getOrder()->getStatus()->getId(), [1]))
            {
                $this->addFlash('notice', 'Le registre a bien été mis à jour sur la commande #'.$accessibilityRegister->getOrder()->getId().'.');

                return $this->redirectToRoute('order_choose', ['order_id' => $accessibilityRegister->getOrder()->getId()]);
            }
        }

        # If order in progress already exist for the company
        $orders = $this->getUser()->getCompany()->getOrders();
        foreach($orders as $single_order)
        {
            if(in_array($single_order->getStatus()->getId(), [1]))
            {

                # If register are already finished (validatedAt)
                if($accessibilityRegister->getValidatedAt()){
                    $this->addFlash('warning', 'Vous ne pouvez pas ajouter ce registre à la commande.');
                    return $this->redirectToRoute('order_choose', ['order_id' => $single_order->getId()]);
                }

                # If ERP have valid subscription
                if($single_order->getAccessibilityRegisters()){
                    foreach($single_order->getAccessibilityRegisters() as $register)
                    {
                        if($register->getErp()->getSubscriptions()){
                            foreach($register->getErp()->getSubscriptions() as $subscription)
                            {
                                if($subscription->isValid()){
                                    $this->addFlash('warning', 'Vous ne pouvez pas ajouter ce registre à la commande car un abonnement est en cours.');
                                    return $this->redirectToRoute('order_choose', ['order_id' => $single_order->getId()]);
                                }
                            }
                        }
                    }
                }


                $accessibilityRegister->setOrder($single_order);
                $em = $this->getDoctrine()->getManager();
                $em->persist($single_order);
                $em->persist($accessibilityRegister);
                $em->flush();

                $this->addFlash('notice', 'Le registre a bien été ajouté à la commande #'.$single_order->getId());

                return $this->redirectToRoute('order_choose', ['order_id' => $single_order->getId()]);

                break;
            }
        }

        # If register are already finished (validatedAt)
        if($accessibilityRegister->getValidatedAt()){
            $this->addFlash('warning', 'Vous ne pouvez pas ajouter ce registre à la commande');
            return $this->redirectToRoute('dashboard_index');
        }

        $order = new Order();

        $status = $this->getDoctrine()->getRepository(RefOrderStatus::class)->findOneBy(
            ['id' => 1]
        );

        $order->setCompany($accessibilityRegister->getErp()->getCompany());
        $order->setUser($this->getUser());
        $order->setStatus($status);

        $accessibilityRegister->setOrder($order);

        $em = $this->getDoctrine()->getManager();
        $em->persist($order);
        $em->persist($accessibilityRegister);
        $em->flush();

        $this->addFlash('notice', 'La commande a bien été créée.');

        $orderHistoryManager->setHistory($order, null, 1, $this->getUser());

        return $this->redirectToRoute('order_choose', ['order_id' => $order->getId()]);
    }


    /**
     * @Route("/dashboard/order/choose/{order_id}", name="order_choose")
     */
    public function orderChooseAction(Request $request, $order_id, AccessManager $accessManager)
    {
        $order = $this->getDoctrine()->getRepository(Order::class)->findOneBy([
            'id' => $order_id
        ]);


        if($this->getUser()->getCompany()->getRole() == 'large'){
            return $this->redirectToRoute('order_choose_large', ['order_id' => $order_id]);
        }

        if(!$accessManager->checkUserbyOrder($this->getUser(), $order))
        {
            throw new AccessDeniedHttpException('Vous ne pouvez pas modifier cette commande.');
        }

        if($order->getStatus()->getId() != 1)
        {
            return $this->redirectByStatus($order);
        }


        $products = $this->getDoctrine()->getRepository(Product::class)->findAll();

        return $this->render('Front/Dashboard/order/choose.html.twig', [
            'order' => $order,
            'products' => $products
        ]);
    }

    /**
     * @Route("/dashboard/order/removeRegister/{order_id}/{register_id}", name="order_remove_register")
     */
    public function orderRemoveRegisterAction(Request $request, $order_id, $register_id, AccessManager $accessManager)
    {
        $order = $this->getDoctrine()->getRepository(Order::class)->findOneBy([
            'id' => $order_id
        ]);

        $register = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findOneBy([
            'id' => $register_id
        ]);

        if(!$accessManager->checkUserbyOrder($this->getUser(), $order))
        {
            throw new AccessDeniedHttpException('Vous ne pouvez pas modifier cette commande.');
        }

        if($order->getStatus()->getId() != 1)
        {
            return $this->redirectByStatus($order);
        }

        if(count($order->getAccessibilityRegisters()) < 2)
        {
            $em = $this->getDoctrine()->getManager();


            if($register->getOrder() == $order)
            {
                $register->setOrder(null);
                $em = $this->getDoctrine()->getManager();
                $em->persist($register);
                $em->flush();
            }

            foreach($order->getOrderDetails() as $detail)
            {
                $em->remove($detail);
                $em->flush();
            }

            foreach($order->getOrderHistorys() as $history)
            {
                $em->remove($history);
                $em->flush();
            }
            $em->remove($order);
            $em->flush();

            return $this->redirectToRoute('dashboard_index');
        }


        if($register->getOrder() == $order)
        {
            $register->setOrder(null);
            $em = $this->getDoctrine()->getManager();
            $em->persist($register);
            $em->flush();
        }

        return $this->redirectToRoute('order_choose', [
            'order_id' => $order->getId()
        ]);
    }

    /**
     * @Route("/dashboard/order/list", name="order_list")
     */
    public function orderListAction(Request $request)
    {
        $orders = $this->getDoctrine()->getRepository(Order::class)->findBy([
            'company' => $this->getUser()->getCompany()
        ], [
            'created_at' => 'DESC'
        ]);


        return $this->render('Front/Dashboard/order/list.html.twig', [
            'orders' => $orders,
        ]);
    }

    /**
     * @Route("/dashboard/order/choose_delivery/{order_id}", name="order_choose_delivery")
     */
    public function orderChooseDeliveryAction(Request $request, $order_id, AccessManager $accessManager)
    {
        $order = $this->getDoctrine()->getRepository(Order::class)->findOneBy([
            'id' => $order_id
        ]);

        if(!$accessManager->checkUserbyOrder($this->getUser(), $order))
        {
            throw new AccessDeniedHttpException('Vous ne pouvez pas voir cette commande.');
        }

        if($order->getStatus()->getId() != 2)
        {
            return $this->redirectByStatus($order);
        }

        if($order->getDeliveryChoice())
        {
            return $this->redirectToRoute('order_pay', [
                'order_id' => $order->getId()
            ]);
        }

        $delivery_address = new DeliveryAddress();
        $form = $this->createForm(DeliveryAddressType::class, $delivery_address);
        $form->handleRequest($request);

        if($request->isMethod('POST'))
        {
            $delivery_choice = $request->request->get('choiceRadios');

            if(in_array($delivery_choice, ['erp', 'company'])) {

                $order->setDeliveryChoice($delivery_choice);
                $em = $this->getDoctrine()->getManager();
                $em->persist($order);
                $em->flush();
                return $this->redirectToRoute('order_pay', [
                    'order_id' => $order->getId()
                ]);

            }else{
                if($form->isSubmitted() && $form->isValid())
                {
                    $address = $request->request->get('delivery_address');
                    $em = $this->getDoctrine()->getManager();
                    $em->persist($delivery_address);
                    $em->flush();

                    $order->setDeliveryAddress($delivery_address);
                    $order->setDeliveryChoice('custom');
                    $em->persist($order);
                    $em->flush();
                    return $this->redirectToRoute('order_pay', [
                        'order_id' => $order->getId()
                    ]);
                }
            }

        }

        return $this->render('Front/Dashboard/order/delivery.html.twig', [
            'order' => $order,
            'form' => $form->createView(),
            'errors' => $form->getErrors()
        ]);
    }

    /**
     * @Route("/dashboard/order/confirm/{order_id}", name="order_confirm")
     */
    public function orderConfirmAction(Request $request, $order_id, AccessManager $accessManager, OrderHistoryManager $orderHistoryManager, SubscriptionManager $subscriptionManager)
    {
        $order = $this->getDoctrine()->getRepository(Order::class)->findOneBy([
            'id' => $order_id
        ]);

        if(!$accessManager->checkUserbyOrder($this->getUser(), $order))
        {
            throw new AccessDeniedHttpException('Vous ne pouvez pas agir sur cette commande.');
        }

        if($order->getStatus()->getId() != 1)
        {
            return $this->redirectByStatus($order);
        }

        $qty = count($order->getAccessibilityRegisters());
        $em = $this->getDoctrine()->getManager();

        if($request->isMethod('POST'))
        {
            $products = $request->request->all();

            foreach($products as $product_id)
            {
                $product = $this->getDoctrine()->getRepository(Product::class)->findOneBy([
                    'id' => $product_id
                ]);

                if($product->getSlug() == 'publication-archivee-en-ligne')
                {
                    throw new AccessDeniedHttpException('Il y a une erreur dans votre commande, merci de recommencer.');
                }

                $detail = new OrderDetails();
                $detail->setOrder($order)
                    ->setProduct($product)
                    ->setQuantity($qty)
                ;
                $em->persist($detail);
                $em->flush();
            }

            # Add product "publication-archivee-en-ligne" if no valid subscription
            if($subscriptionManager->checkValidityByOrder($order) == false){
                $product_publication = $this->getDoctrine()->getRepository(Product::class)->findOneBy([
                    'slug' => 'publication-archivee-en-ligne'
                ]);
                $detail_publication = new OrderDetails();
                $detail_publication->setOrder($order)
                    ->setProduct($product_publication)
                    ->setQuantity($qty)
                ;
                $em->persist($detail_publication);
                $em->flush();
            }
        }
        $status = $this->getDoctrine()->getRepository(RefOrderStatus::class)->findOneBy(
            ['id' => 2]
        );

        $orderDetails = $this->getDoctrine()->getRepository(OrderDetails::class)->findBy(
            ['order' => $order]
        );

        $priceHt = [];
        $amountTva = [];
        foreach($orderDetails as $detail)
        {
            $priceHt[] = $detail->getProduct()->getPriceHt() * $detail->getQuantity();
            $amountTva[] = ($detail->getProduct()->getPriceHt() * $detail->getProduct()->getTva()->getRate()) * $detail->getQuantity();
        }

        $priceHt = array_sum($priceHt);
        $amountTva = array_sum($amountTva);
        $priceTtc = $priceHt + $amountTva;

        $order->setStatus($status)
            ->setAmountHt($priceHt)
            ->setAmountTva($amountTva)
            ->setAmountTtc($priceTtc)
        ;

        $em->persist($order);
        $em->flush();

        $orderHistoryManager->setHistory($order, 1, 2, $this->getUser());

        # Check if one of products are affected by delivery
        foreach($order->getOrderDetails() as $single_product){

            $product = $single_product->getProduct();

            if(in_array($product->getSlug(), ['impression-papier','publication-archivee-en-ligne']))
            {
                if(!$order->getDeliveryChoice())
                {
                    return $this->redirectToRoute('order_choose_delivery', [
                        'order_id' => $order->getId()
                    ]);
                }
                break;
            }
        }

        return $this->redirectToRoute('order_pay', [
        'order_id' => $order_id
    ]);
    }

    /**
     * @Route("/dashboard/order/goback/{order_id}", name="order_goback")
     */
    public function orderGoBackAction(Request $request, $order_id, AccessManager $accessManager, OrderHistoryManager $orderHistoryManager)
    {
        $order = $this->getDoctrine()->getRepository(Order::class)->findOneBy([
            'id' => $order_id
        ]);

        if(!$accessManager->checkUserbyOrder($this->getUser(), $order))
        {
            throw new AccessDeniedHttpException('Vous ne pouvez pas voir cette commande.');
        }

        if($order->getStatus()->getId() != 2)
        {
            return $this->redirectByStatus($order);
        }

        $details = $order->getOrderDetails();
        $em = $this->getDoctrine()->getManager();

        foreach($details as $detail)
        {
            $em->remove($detail);
            $em->flush();
        }

        $order->setDeliveryChoice(null);
        $order->setDeliveryAddress(null);
        $status = $this->getDoctrine()->getRepository(RefOrderStatus::class)->findOneBy(
            ['id' => 1]
        );

        $order->setStatus($status);
        $em->persist($order);
        $em->flush();

        $orderHistoryManager->setHistory($order, 2, 1, $this->getUser());

        return $this->redirectToRoute('order_choose', [
            'order_id' => $order_id
        ]);

    }

    /**
     * @Route("/dashboard/order/pay/{order_id}", name="order_pay")
     */
    public function orderPayAction(Request $request, $order_id, AccessManager $accessManager)
    {
        $order = $this->getDoctrine()->getRepository(Order::class)->findOneBy([
            'id' => $order_id
        ]);

        if(!$accessManager->checkUserbyOrder($this->getUser(), $order))
        {
            throw new AccessDeniedHttpException('Vous ne pouvez pas voir cette commande.');
        }

        if($order->getStatus()->getId() != 2)
        {
            return $this->redirectByStatus($order);
        }

        # PROCESS TO PAYEMENT

        $paybox = $this->get('lexik_paybox.request_handler');
        $paybox->setParameters(array(
            'PBX_CMD'          => $order->getId(),
            'PBX_DEVISE'       => '978',
            'PBX_PORTEUR'      => $order->getUser()->getEmail(),
            'PBX_RETOUR'       => 'id:R;authorization:A;result:E;transaction:S;amount:M;date:W;time:Q;card_type:C;card_end:D;card_print:H',
            'PBX_TOTAL'        => $order->getAmountTtc() * 100,
            'PBX_TYPEPAIEMENT' => 'CARTE',
            'PBX_TYPECARTE'    => 'CB',
            'PBX_EFFECTUE'     => $this->generateUrl('order_pay_return', array('status' => 'success'), UrlGeneratorInterface::ABSOLUTE_URL),
            'PBX_REFUSE'       => $this->generateUrl('order_pay_return', array('status' => 'denied'), UrlGeneratorInterface::ABSOLUTE_URL),
            'PBX_ANNULE'       => $this->generateUrl('order_pay_return', array('status' => 'canceled'), UrlGeneratorInterface::ABSOLUTE_URL),
            'PBX_RUF1'         => 'GET',
            'PBX_REPONDRE_A'   => $this->generateUrl('lexik_paybox_ipn', array('time' => time()), UrlGeneratorInterface::ABSOLUTE_URL),
        ));

        # END OF PROCESS

        return $this->render('Front/Dashboard/order/pay.html.twig', [
            'order' => $order,
            'url' => $paybox->getUrl(),
            'form' => $paybox->getForm()->createView()
        ]);
    }

    /**
     * @Route("/dashboard/order/pay/return/{status}", name="order_pay_return")
     */
    public function orderPayReturnAction(Request $request, $status, AccessManager $accessManager)
    {

        $order_id = $request->query->get('id');

        $order = $this->getDoctrine()->getRepository(Order::class)->findOneBy([
            'id' => $order_id
        ]);

        if(!$accessManager->checkUserbyOrder($this->getUser(), $order))
        {
            throw new AccessDeniedHttpException('Vous ne pouvez pas voir cette commande.');
        }

        if($order->getStatus()->getId() == 3)
        {
            return $this->redirectToRoute('order_pay_confirm', ['order_id' => $order->getId()]);
        }

        return $this->render(
            'Front/Dashboard/order/return.html.twig',
            array(
                'status'     => $status,
                'parameters' => $request->query,
                'order' => $order
            )
        );
    }

    /**
     * @Route("/dashboard/order/pay/confirm/{order_id}", name="order_pay_confirm")
     */
    public function orderPayConfirmAction(Request $request, $order_id, AccessManager $accessManager, PdfGenerator $pdfGenerator, OrderHistoryManager $orderHistoryManager, InvoiceGenerator $invoiceGenerator, MailManager $mailManager, SubscriptionManager $subscriptionManager)
    {
        $order = $this->getDoctrine()->getRepository(Order::class)->findOneBy([
            'id' => $order_id
        ]);

        if(!$accessManager->checkUserbyOrder($this->getUser(), $order))
        {
            throw new AccessDeniedHttpException('Vous ne pouvez pas voir cette commande.');
        }

        if($order->getStatus()->getId() != 3)
        {
            if(!is_null($order->getValidatedAt()))
            {
                return $this->redirectToRoute('dashboard_index');
            }

            return $this->redirectByStatus($order);
        }

        if($order->getValidatedAt())
        {
            $this->addFlash('notice', 'Votre commande a déjà été validée.');
            return $this->redirectToRoute('order_summary', [
                'order_id' => $order->getId()
            ]);
        }

        $details = $order->getOrderDetails();

        $registers = $order->getAccessibilityRegisters();

        foreach ($details as $detail)
        {

            if(in_array($detail->getProduct()->getSlug(), ['impression-papier'])){
                foreach($registers as $register)
                {
                    $refPrintStatus = $this->getDoctrine()->getRepository(RefPrintStatus::class)->findOneBy(['id' => 1]);
                    $printRequest = new PrintRequest();
                    $printRequest
                        ->setOrder($order)
                        ->setRefPrintStatus($refPrintStatus)
                        ->setRegister($register);

                    $em = $this->getDoctrine()->getManager();
                    $em->persist($printRequest);
                    $em->flush();
                }
            }
            if($detail->getProduct()->getSlug() == 'publication-archivee-en-ligne'){

                foreach($registers as $register)
                {
                    # Attribute QrCode to order
                    $qrCode = $this->getDoctrine()->getRepository(\AppBundle\Entity\QrCode::class)->findByAvailableQrCode();
                    if($qrCode) {
                        $em = $this->getDoctrine()->getManager();
                        $register->setQrCode($qrCode);
                        $em->persist($register);

                        $qrCode->setAssignedAt(new \DateTime());

                        $em->persist($qrCode);
                        $em->flush();
                    }

                    $qrCodeRequest = new QrCodeRequest();
                    $qrCodeRequest->setStatus('pending')
                        ->setRegister($register)
                        ->setOrder($order);

                    $em = $this->getDoctrine()->getManager();
                    $em->persist($qrCodeRequest);
                    $em->flush();
                }
            }
            if($detail->getProduct()->getSlug() == 'verification-par-mydl'){
                foreach($registers as $register)
                {
                    $em = $this->getDoctrine()->getManager();
                    $register->setValidationStatus('pending');
                    $em->persist($register);
                    $em->flush();
                }
            }
        }

        # Generate Invoice for order
        $invoiceGenerator->generate($order);
        # Send order confirmation email
        $mailManager->sendOrderConfirmationEmail($order);
        $mailManager->sendOrderValidatedNotificationEmail($order);

        $order->setValidatedAt(new \DateTime());

        $orderHistoryManager->setHistory($order, 2, 3, $this->getUser());


        # Is no valid Subscription
        if($subscriptionManager->checkValidityByOrder($order) == false){
            $dateEnd = new \DateTime();
            $dateEnd->modify('+1 year');

            foreach($registers as $register)
            {
                $subscription = new Subscription();
                $subscription
                    ->setOrder($order)
                    ->setUser($this->getUser())
                    ->setErp($register->getErp())
                    ->setDateStart(new \DateTime())
                    ->setDateEnd($dateEnd);

                $em = $this->getDoctrine()->getManager();
                $em->persist($subscription);
                $em->flush();
            }
        }

        foreach($registers as $register)
        {
            $register->setValidatedAt(new \DateTime());
            $em = $this->getDoctrine()->getManager();
            $em->persist($register);
            $em->flush();
        }


        $em = $this->getDoctrine()->getManager();
        $em->persist($order);
        $em->persist($register);
        $em->flush();

        # If this is the first order, generate Parrainage code
        $orders = $this->getUser()->getCompany()->getOrders();
        if(count($orders) == 1){
            $chars = "ABCDEFGHKMNPRSTUVWXYZ23456789";
            $code = substr(str_shuffle($chars), 0, 8);
            $company = $this->getUser()->getCompany();
            $company->setCodeParrainage($code);
            $this->addFlash('success', 'Votre code parrainage a été généré, transmettez-le à des filleuls pour bénéficier d\'un an d\'abonnement gratuit.');
            $em->persist($company);
            $em->flush();

            # if user company have parrain
            if($this->getUser()->getCompany()->getParrain()){
                $parrain = $this->getUser()->getCompany()->getParrain();
                $bonus = new BonusParrainage();
                $bonus->setParrain($parrain)
                    ->setFilleul($this->getUser()->getCompany());

                $em->persist($bonus);
                $em->flush();

                $mailManager->sendBonusParrainageEmail($parrain, $this->getUser());
            }

        }

        $this->addFlash('notice', 'Votre commande a bien été validée, vous avez désormais accès aux fiches des Registres d\'Accessibilité et à votre facture.');

        return $this->redirectToRoute('order_summary', [
            'order_id' => $order->getId(),
        ]);
    }

    /**
     * @Route("/dashboard/order/summary/{order_id}", name="order_summary")
     */
    public function orderSummaryAction(Request $request, $order_id, AccessManager $accessManager)
    {
        $order = $this->getDoctrine()->getRepository(Order::class)->findOneBy([
            'id' => $order_id
        ]);

        if(!$accessManager->checkUserbyOrder($this->getUser(), $order))
        {
            throw new AccessDeniedHttpException('Vous ne pouvez pas voir le récapitulatif de cette commande.');
        }

        if(!in_array($order->getStatus()->getId(), [3]))
        {
            return $this->redirectByStatus($order);
        }

        if(is_null($order->getValidatedAt()))
        {
            return $this->redirectToRoute('order_pay_confirm');
        }

        return $this->render('Front/Dashboard/order/summary.html.twig', [
            'order' => $order,
        ]);
    }

    /**
     * @Route("/dashboard/order/invoice/download/{order_id}", name="order_invoice_download")
     */
    public function orderInvoiceDownloadAction($order_id, AccessManager $accessManager)
    {
        $order = $this->getDoctrine()->getRepository(Order::class)->findOneBy([
            'id' => $order_id
        ]);

        if(!$accessManager->checkUserbyOrder($this->getUser(), $order))
        {
            throw new AccessDeniedHttpException("Vous n'avez pas l'autorisation de télécharger ce fichier.");
        }

        $response = new BinaryFileResponse('invoices/'.$order->getInvoice()->getPdf());
        $response->setContentDisposition(ResponseHeaderBag::DISPOSITION_ATTACHMENT);

        return $response;
    }

    /**
     * @Route("/dashboard/order/direct/{register_id}", name="order_direct")
     */
    public function orderDirectAction($register_id, AccessManager $accessManager)
    {
        $register = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findOneBy([
            'id' => $register_id
        ]);

        if(!$accessManager->checkUserbyRegister($this->getUser(), $register))
        {
            throw new AccessDeniedHttpException("Vous n'avez pas l'autorisation de crée cette commande.");
        }

        $order = new Order();
        $order_detail = new OrderDetails();
        $new_register = new AccessibilityRegister();

        $files = $register->getFileUploads();
        foreach($files as $file)
        {
            $new_register->addFileUpload($file);
        }

        $new_register->setOrder($order)
            ->setCreatedAt(new \DateTime())
            ->setErp($register->getErp())
        ;

        $status = $this->getDoctrine()->getRepository(RefOrderStatus::class)->findOneBy(['id' => 2]);

        $product = $this->getDoctrine()->getRepository(Product::class)->findOneBy([
            'slug' => 'impression-papier'
        ]);

        $amountTva = $product->getPriceHt() * $product->getTva()->getRate();

        $order->setCompany($register->getErp()->getCompany())
            ->setStatus($status)
            ->setUser($this->getUser())
            ->setDeliveryChoice('erp')
            ->setAmountHt($product->getPriceHt())
            ->setAmountTva($amountTva)
            ->setAmountTtc($amountTva + $product->getPriceHt());

        $order_detail->setOrder($order)
            ->setCreatedAt(new \DateTime())
            ->setProduct($product)
            ->setQuantity(1)
        ;

        $em = $this->getDoctrine()->getManager();
        $em->persist($order);
        $em->persist($new_register);
        $em->persist($order_detail);
        $em->flush();

        return $this->redirectToRoute('order_pay', ['order_id' => $order->getId()]);

    }

}
