<?php

namespace FrontBundle\Controller;

use AppBundle\Entity\AccessibilityRegister;
use AppBundle\Entity\Erp;
use AppBundle\Entity\FileUpload;
use AppBundle\Entity\Order;
use AppBundle\Entity\OrderDetails;
use AppBundle\Entity\OrderHistory;
use AppBundle\Entity\Product;
use AppBundle\Entity\RefOrderStatus;
use AppBundle\Entity\RefTypeFile;
use AppBundle\Service\AccessManager;
use AppBundle\Service\FileUploader;
use AppBundle\Service\OrderHistoryManager;
use AppBundle\Service\PdfGenerator;
use Spipu\Html2Pdf\Html2Pdf;
use Symfony\Component\Config\Definition\Exception\Exception;
use FrontBundle\Form\AccessibilityRegisterType;
use FrontBundle\Form\FileUploadType;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\Validator\Constraints\DateTime;

class AccessibilityRegisterController extends Controller
{
    /**
     * @Route("/dashboard/register/create/{erp_id}", name="register_create", host="%domain%")
     */
    public function registerCreateAction(Request $request, $erp_id, AccessManager $accessManager)
    {

        $erp = $this->getDoctrine()->getRepository(Erp::class)->findOneBy(
            ['id' => $erp_id]
        );

        if(!$accessManager->checkUserbyErp($this->getUser(), $erp))
        {
            throw new AccessDeniedHttpException('Vous ne pouvez pas créer ce registre.');
        }

        $parent_register = null;
        # Check if register are pending status validation
        $lastOrdinanceRegister = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findOneByLastValidationRegister($erp);
        if($lastOrdinanceRegister){
            if($lastOrdinanceRegister->getValidationStatus() == 'pending'){
                $this->addFlash('warning', "Vous ne pouvez pas modifier votre registre car celui-ci est actuellement vérifié par nos équipes.");
                return $this->redirectToRoute('erp_show', ['erp_id' => $erp->getId()]);
            }elseif($lastOrdinanceRegister->getValidationStatus() == 'ko'){
                if($lastOrdinanceRegister->getParent()){
                    $parent_register = $lastOrdinanceRegister->getParent();
                }else{
                    $parent_register = $lastOrdinanceRegister;
                }
            }
        }

        $last_register = null;

        # Check if others registers un-ordered exists
        if($erp->getAccessibilityRegisters()) {
            # Check if others registers un-validated exists
            foreach($erp->getAccessibilityRegisters() as $register)
            {
                if(!$register->getValidatedAt()){
                    return $this->redirectToRoute('register_edit', ['register_id' => $register->getId()]);
                }
            }

            # Get last register
            $last_register = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findOneBy(
                ['erp' => $erp],
                ['validated_at' => 'DESC']
            );
        }

        $register = new AccessibilityRegister();
        $register->setErp($erp);
        $register->setParent($parent_register);


        # Add all relations (File Uploads) of last register to keep medias
        if($last_register) {
            $last_files = $last_register->getFileUploads();
            foreach($last_files as $file)
            {
                $register->addFileUpload($file);
            }
        }else{
            $fileType = $this->getDoctrine()->getRepository(RefTypeFile::class)->findOneBy(['id' => 9]);
            $fasciculeAidePersonne = new FileUpload();
            $fasciculeAidePersonne
                ->setErp($erp)
                ->setName('bien-accueillir-les-personnes-handicapees.pdf')
                ->setPath('bien-accueillir-les-personnes-handicapees.pdf')
                ->setOrder(1)
                ->setFileType($fileType)
                ->addAccessibilityRegister($register);
            $em = $this->getDoctrine()->getManager();
            $em->persist($fasciculeAidePersonne);
            $em->flush();

            $register->addFileUpload($fasciculeAidePersonne);
        }

        $em = $this->getDoctrine()->getManager();
        $em->persist($register);
        $em->flush();

        return $this->redirectToRoute('register_edit', [
            'register_id' => $register->getId()
        ]);
    }

    /**
     * @Route("/dashboard/register/new_correction/{register_id}", name="register_new_correction", host="%domain%")
     */
    public function accessibilityRegisterNewCorrectionAction($register_id, AccessManager $accessManager, OrderHistoryManager $orderHistoryManager)
    {
        $register = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findOneBy([
            'id' => $register_id
        ]);

        if(!$accessManager->checkUserbyRegister($this->getUser(), $register))
        {
            throw new AccessDeniedHttpException("Vous n'avez pas l'autorisation d'accéder à ce registre.");
        }

        $status = $this->getDoctrine()->getRepository(RefOrderStatus::class)->findOneBy(
            ['id' => 2]
        );

        $product = $this->getDoctrine()->getRepository(Product::class)->findOneBy(
            ['slug' => 'verification-par-mydl']
        );

        $em = $this->getDoctrine()->getManager();

        $order = new Order();
        $order
            ->setCompany($register->getErp()->getCompany())
            ->setUser($this->getUser())
            ->setStatus($status)
        ;

        $detail = new OrderDetails();
        $detail->setOrder($order)
            ->setProduct($product)
            ->setQuantity(1)
        ;
        $em->persist($detail);
        $em->flush();

        $orderDetails = $this->getDoctrine()->getRepository(OrderDetails::class)->findBy(
            ['order' => $order]
        );

        $priceHt = [];
        $amountTva = [];
        foreach($orderDetails as $detail)
        {
            $priceHt[] = $detail->getProduct()->getPriceHt() * $detail->getQuantity();
            $amountTva[] = ($detail->getProduct()->getPriceHt() * $detail->getProduct()->getTva()->getRate()) * $detail->getQuantity();
        }

        $priceHt = array_sum($priceHt);
        $amountTva = array_sum($amountTva);
        $priceTtc = $priceHt + $amountTva;

        $order->setStatus($status)
            ->setAmountHt($priceHt)
            ->setAmountTva($amountTva)
            ->setAmountTtc($priceTtc)
        ;

        $register->setOrder($order);
        $em->persist($order);
        $em->persist($register);
        $em->flush();

        $orderHistoryManager->setHistory($order, 1, 2, $this->getUser());

        return $this->redirectToRoute('order_pay', [
            'order_id' => $order->getId()
        ]);
    }

    /**
     * @Route("/dashboard/register/update/{register_id}", name="register_update", host="%domain%")
     */
    public function accessibilityRegisterUpdateAction($register_id, AccessManager $accessManager)
    {
        $register = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findOneBy([
            'id' => $register_id
        ]);

        if(!$accessManager->checkUserbyRegister($this->getUser(), $register))
        {
            throw new AccessDeniedHttpException("Vous n'avez pas l'autorisation d'accéder à ce registre.");
        }

        $subscriptionValid = false;
        # Check if erp subscription is valid
        if($register->getErp()->getSubscriptions()){
            foreach($register->getErp()->getSubscriptions() as $subscription)
            {
                if($subscription->isValid())
                {
                    $subscriptionValid = true;
                    break;
                }
            }
        }else{
            $this->addFlash('notice', "Vous ne pouvez pas mettre à jour ce registre car vous ne possédez pas d'abonnement.");
        }

        # Check if register is already validated
        if($register->getValidatedAt()){
            $this->addFlash('notice', "Vous ne pouvez pas mettre à jour ce registre.");
            return $this->redirectToRoute('dashboard_index');
        }


        if($subscriptionValid){
            $register->setValidatedAt(new \DateTime());

            # Check if register parent have PrintRequest
            if($register->getParent()){
                $parent_register = $register->getParent();
                if($parent_register->getPrintRequest()){

                    $printRequest = $parent_register->getPrintRequest();
                    $printRequest->setRegister($register);

                    $em = $this->getDoctrine()->getManager();
                    $em->persist($printRequest);
                    $em->flush();
                }
            }

            $em = $this->getDoctrine()->getManager();
            $em->persist($register);
            $em->flush();

            return $this->redirectToRoute('erp_show', [
                'erp_id' => $register->getErp()->getId()
            ]);
        }else{
            $this->addFlash('notice', "Votre abonnement est expiré.");
            return $this->redirectToRoute('dashboard_index');
        }


    }

    /**
     * @Route("/dashboard/register/edit/{register_id}", name="register_edit", host="%domain%")
     */
    public function registerEditAction(Request $request, $register_id, FileUploader $fileUploader, AccessManager $accessManager)
    {

        $register = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findOneBy(
            ['id' => $register_id]
        );

        if(!$accessManager->checkUserbyRegister($this->getUser(), $register))
        {
            throw new AccessDeniedHttpException('Vous n\'avez pas accès à ce registre.');
        }

        # If register order is complete
        if($register->getOrder())
        {
            if(in_array($register->getOrder()->getStatus()->getId(), [3]))
            {
                return $this->redirectToRoute('register_create', ['erp_id' => $register->getErp()->getId()]);
            }
        }


        $forms = [];
        $fileUploads = [];
        $types = $register->getErp()->getCategoryErp()->getTypesFile();
        $fileType = null;
        $formFile = null;
        foreach($types as $type) {
            $fileUploads[$type->getId()] = $register->getFileUploadsByTypeId($type->getId());
            $fileUpload = new FileUpload();
            $fileUpload->addAccessibilityRegister($register);
            $fileUpload->setOrder(10);
            $fileUpload->setErp($register->getErp());
            $fileUpload->setFileType($type);
            $form = $this->createForm(FileUploadType::class, $fileUpload, ['type' => $type, 'types' => $types]);
            if(!is_null($request->request->get('file_upload')) && $request->request->get('file_upload')['file_type_id'] == $type->getId())
            {
                $fileType = $type;
                $formFile = $form;
            }
            $forms[$type->getId()] = $form->createView();
        }


        if($request->isMethod('POST'))
        {
            $form = $formFile;


            $file = $request->files->get('file_upload')['file_uploaded'];

            # pour tester la taille : if($file->getSize())
            if($file->getMimeType() == 'application/pdf' && $file->getSize() < 8500000)
            {

                $fileName = $fileUploader->upload($file);

                $datas = $request->request->get('file_upload');
                $datas['name'] = $file->getClientOriginalName();
                $datas['path'] = $fileName;
                $request->request->set('file_upload', $datas);

                $form->handleRequest($request);

                if ($form->isSubmitted() && $form->isValid())
                {

                    $em = $this->getDoctrine()->getManager();
                    $register->setUpdatedAt(new \DateTime());
                    $register->addFileUpload($form->getData());
                    $em->persist($form->getData());
                    $em->persist($register);
                    $em->flush();

                    return $this->redirectToRoute('register_edit', [
                        'register_id' => $register->getId()
                    ]);
                }
            }
            else
            {
                $this->addFlash('danger', 'Le fichier n\'est pas un PDF.');
            }
        }


        # Get All registers of ERP by date DESC
        $registers = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findBy(['erp' => $register->getErp()],[
            'validated_at' => 'DESC'
        ]);

        return $this->render('Front/Dashboard/register/edit.html.twig', [
            'linked_erp' => $register->getErp(),
            'forms' => $forms,
            'types' => $types,
            'fileUploads' => $fileUploads,
            'register' => $register,
            'registers' => $registers
        ]);
    }

    /**
     * @Route("/dashboard/AccessibilityRegister/list", name="register_list")
     */
    public function accessibilityRegisterListAction()
    {
        if(!$this->getUser()->getCompany()->getRole() == 'large'){
            return $this->redirectToRoute('dashboard_index');
        }

        $erps = $this->getDoctrine()->getRepository(Erp::class)->findBy([
            'company' => $this->getUser()->getCompany()
        ]);

        $registers = [];

        foreach($erps as $erp)
        {
            $registers[] = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findOnePublicAndAuthorisedPdfVersionRegister($erp);
        }

        return $this->render('Front/Dashboard/register/list.html.twig', [
            'registers' => $registers
        ]);
    }

    /**
     * @Route("/dashboard/AccessibilityRegister/generate/{register_id}", name="register_generate")
     */
    public function accessibilityRegisterGenerateAction(Request $request, $register_id, AccessManager $accessManager, PdfGenerator $pdfGenerator)
    {
        if(!$this->getUser()->getCompany()->getRole() == 'large'){
            return $this->redirectToRoute('dashboard_index');
        }

        $register = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findOneBy(
            ['id' => $register_id]
        );

        if(!$accessManager->checkUserbyRegister($this->getUser(), $register))
        {
            throw new AccessDeniedHttpException('Vous ne pouvez pas générer ce registre.');
        }

        $types = $register->getErp()->getCategoryErp()->getTypesFile();
        $fileUploads = [];
        foreach($types as $type) {
            $fileUploads[$type->getId()] = $register->getFileUploadsByTypeId($type->getId());
        }

        if($register->getPdf()){
            $this->addFlash('notice', 'Le registre a déjà été généré.');
            return $this->redirectToRoute('dashboard_index');
        }

        $pdfGenerator->generateOnePdf($register);


        $response = new BinaryFileResponse('uploads/'.$register->getPdf());
        $response->setContentDisposition(ResponseHeaderBag::DISPOSITION_ATTACHMENT);
        return $response;
    }

    /**
     * @Route("/dashboard/register/download/{register_id}", name="register_download")
     */
    public function registerDownloadAction(Request $request, $register_id, AccessManager $accessManager)
    {

        if(!$this->getUser()->getCompany()->getRole() == 'large'){
            return $this->redirectToRoute('dashboard_index');
        }

        $register = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findOneBy(
            ['id' => $register_id]
        );

        if(!$accessManager->checkUserbyRegister($this->getUser(), $register))
        {
            throw new AccessDeniedHttpException('Vous ne pouvez pas générer ce registre.');
        }

        $response = new BinaryFileResponse('uploads/'.$register->getPdf());
        $response->setContentDisposition(ResponseHeaderBag::DISPOSITION_ATTACHMENT);
        return $response;
    }

    /**
     * @Route("/dashboard/AccessibilityRegister/view/{accessibility_register_id}", name="register_view", host="%domain%")
     */
    public function accessibilityRegisterViewAction($accessibility_register_id, AccessManager $accessManager)
    {
        $accessibility_register = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findOneBy([
            'id' => $accessibility_register_id
        ]);

        if(!$accessManager->checkUserbyRegister($this->getUser(), $accessibility_register))
        {
            throw new AccessDeniedHttpException("Vous n'avez pas l'autorisation d'accéder à ce fichier.");
        }

        return new BinaryFileResponse('uploads/'.$accessibility_register->getPdf());
    }

    /**
     * @Route("/dashboard/qrcode/view/{accessibility_register_id}", name="qrcode_view", host="%domain%")
     */
    public function qrCodeViewAction($accessibility_register_id)
    {

        $accessibilityRegister = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findOneBy([
            'id' => $accessibility_register_id
        ]);

        return new BinaryFileResponse('qrcodes/pdf/'.$accessibilityRegister->getQrCode()->getPdf());
    }

    /**
     * @Route("/dashboard/qrcode/download/{accessibility_register_id}", name="qrcode_download", host="%domain%")
     */
    public function qrCodeDownloadAction($accessibility_register_id)
    {

        $accessibilityRegister = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findOneBy([
            'id' => $accessibility_register_id
        ]);
        $response = new BinaryFileResponse('qrcodes/pdf/'.$accessibilityRegister->getQrCode()->getPdf());
        $response->setContentDisposition(ResponseHeaderBag::DISPOSITION_ATTACHMENT);
        return $response;
    }

    /**
     * @Route("/printer/{accessibilityRegister_token}", name="register_printer_download", host="%domain%")
     */
    public function accessibilityRegisterPrinterDownloadAction($accessibilityRegister_token)
    {

        $accessibiltyRegister = $this->getDoctrine()->getRepository(AccessibilityRegister::class)->findOneBy([
            'token' => $accessibilityRegister_token
        ]);

        if(!$accessibiltyRegister)
        {
            return $this->redirectToRoute('homepage');
        }

        $accessibiltyRegister->setToken(null);
        $em = $this->getDoctrine()->getManager();
        $em->persist($accessibiltyRegister);
        $em->flush();

        $response = new BinaryFileResponse('uploads/'.$accessibiltyRegister->getPdf());
        $response->setContentDisposition(ResponseHeaderBag::DISPOSITION_ATTACHMENT);

        return $response;
    }

}
