<?php

namespace AppBundle\Service;

use AppBundle\Entity\AccessibilityRegister;
use AppBundle\Entity\Company;
use AppBundle\Entity\Exchange;
use AppBundle\Entity\Mail;
use AppBundle\Entity\Order;
use AppBundle\Entity\RefPrinter;
use AppBundle\Entity\User;

class MailManager
{

    protected $mailer;
    protected $templating;
    protected $mailFrom;
    protected $mailFromNom;

    public function __construct(\Swift_Mailer $mailer, $templating, $mailFrom, $mailFromNom)
    {
        $this->mailer = $mailer;
        $this->templating = $templating;
        $this->mailFrom = $mailFrom;
        $this->mailFromNom = $mailFromNom;
    }

    ### -- Connection -- ###

    public function sendUserForgotPasswordEmail(User $user, $token)
    {

        $body = $this->templating->render('emails/forgot_password.html.twig', [
            'token' => $token,
            'username' => $user->getEmail()
        ]);

        $message = (new \Swift_Message())
            ->setSubject('Ré-initialisation de mot de passe')
            ->setFrom($this->mailFrom, $this->mailFromNom)
            ->setTo($user->getEmail())
            ->setBody($body, 'text/html');
        $this->mailer->send($message);
    }

    ### -- Registration -- ###

    public function sendRegistrationNotificationEmail(User $user)
    {
        $body = $this->templating->render('emails/registration_notification.html.twig', [
            'user' => $user
        ]);

        $message = (new \Swift_Message())
            ->setSubject("[Registre d'Accessibilité] Inscription d'un nouvel utilisateur")
            ->setFrom($this->mailFrom, $this->mailFromNom)
            ->setTo(['contact@mydl.fr', 'audrey.stephant@mydl.fr'])
            ->setBody($body, 'text/html');
        $this->mailer->send($message);
    }

    public function sendUserConfirmationEmail(User $user, $token)
    {

        $body = $this->templating->render('emails/confirm_account.html.twig', [
            'token' => $token,
            'username' => $user->getEmail()
        ]);

        $message = (new \Swift_Message())
            ->setSubject('Confirmation de votre adresse e-mail')
            ->setFrom($this->mailFrom, $this->mailFromNom)
            ->setTo($user->getEmail())
            ->setBody($body, 'text/html');
        $this->mailer->send($message);
    }

    public function sendBonusParrainageEmail(Company $parrain, User $filleul)
    {
        $body = $this->templating->render('emails/parrainage_bonus.html.twig', [
            'filleul' => $filleul
        ]);

        $email_parrain = null;
        foreach($parrain->getUsers() as $user)
        {
            $email_parrain = $user->getEmail();
            break;
        }

        $message = (new \Swift_Message())
            ->setSubject('Votre filleul '.$filleul->getFirstName().' '.$filleul->getLastName().' a commandé son Registre d\'Accessibilité.')
            ->setFrom($this->mailFrom, $this->mailFromNom)
            ->setTo($email_parrain)
            ->setBody($body, 'text/html');
        $this->mailer->send($message);
    }

    public function sendParrainageEmail($email, User $user)
    {
        $body = $this->templating->render('emails/parrainage_send.html.twig', [
            'user' => $user
        ]);

        $message = (new \Swift_Message())
            ->setSubject($user->getFirstName().' '.$user->getLastName().' vous invite à créer votre Registre d\'Accessibilité.')
            ->setFrom($this->mailFrom, $this->mailFromNom)
            ->setTo($email)
            ->setBody($body, 'text/html');
        $this->mailer->send($message);
    }

    ### -- Contact form -- ###

    public function sendErpContactEmail(User $user, $erp, $datas)
    {
        $body = $this->templating->render('emails/erp_contact.html.twig', [
            'datas' => $datas,
            'erp' => $erp
        ]);

        $message = (new \Swift_Message())
            ->setSubject('Contact depuis le formulaire "Site Accessible"')
            ->setFrom($this->mailFrom, $this->mailFromNom)
            ->setTo($user->getEmail())
            ->setReplyTo($datas['email'])
            ->setBody($body, 'text/html');
        $this->mailer->send($message);
    }

    public function sendContactEmail($datas)
    {
        $body = $this->templating->render('emails/contact.html.twig', [
            'datas' => $datas,
        ]);

        $message = (new \Swift_Message())
            ->setSubject('Contact depuis le formulaire "Registre d\'Accessibilité"')
            ->setFrom($this->mailFrom, $this->mailFromNom)
            ->setTo('audrey.stephant@mydl.fr')
            ->setReplyTo($datas['email'])
            ->setBody($body, 'text/html');
        if($datas['file']) {
            $message->attach(\Swift_Attachment::fromPath($datas['file']->getPathName())->setFilename($datas['file']->getClientOriginalName()));
        }
        $this->mailer->send($message);
    }

    public function sendCustomEmails(Mail $mail)
    {
        $companies = $mail->getCompanies();
        foreach($companies as $company)
        {
            foreach($company->getUsers() as $user)
            {
                $body = $this->templating->render('emails/custom.html.twig', [
                    'content' => $mail
                ]);

                $message = (new \Swift_Message())
                    ->setSubject($mail->getSubject())
                    ->setFrom($this->mailFrom, $this->mailFromNom)
                    ->setTo($user->getEmail())
                    ->setBody($body, 'text/html');
                $this->mailer->send($message);
            }
        }
    }

    ### -- Order -- ###

    public function sendOrderConfirmationEmail($order)
    {
        $body = $this->templating->render('emails/order_confirmation.html.twig', [
            'order' => $order,
        ]);

        $message = (new \Swift_Message())
            ->setSubject("Confirmation de votre commande - Registre d'Accessibilité")
            ->setFrom($this->mailFrom, $this->mailFromNom)
            ->setTo($order->getUser()->getEmail())
            ->setBody($body, 'text/html');
        $this->mailer->send($message);
    }

    public function sendOrderValidatedNotificationEmail($order)
    {
        $body = $this->templating->render('emails/order_validate_notification.html.twig', [
            'user' => $order->getUser()
        ]);

        $message = (new \Swift_Message())
            ->setSubject("[Registre d'Accessibilité] Nouvelle commande validée")
            ->setFrom($this->mailFrom, $this->mailFromNom)
            ->setTo(['contact@mydl.fr', 'audrey.stephant@mydl.fr'])
            ->setBody($body, 'text/html');
        $this->mailer->send($message);
    }

    public function sendOrderRelanceEmail(Order $order)
    {
        $body = $this->templating->render('emails/order_relance.html.twig', [
            'user' => $order->getUser()
        ]);

        $message = (new \Swift_Message())
            ->setSubject("Votre commande sur Registre d'Accessibilité")
            ->setFrom($this->mailFrom, $this->mailFromNom)
            ->setTo($order->getUser()->getEmail())
            ->setBody($body, 'text/html');
        $this->mailer->send($message);
    }

    ### -- Demand / Request -- ###

    public function sendQrCodePrintConfirmationEmail(Order $order)
    {

        $body = $this->templating->render('emails/qrcode_print_confirmation.html.twig', [
            'order' => $order
        ]);

        $message = (new \Swift_Message())
            ->setSubject('L\'impression de votre QR Code')
            ->setFrom($this->mailFrom, $this->mailFromNom)
            ->setTo($order->getUser()->getEmail())
            ->setBody($body, 'text/html');
        $this->mailer->send($message);
    }

    public function sendPrintConfirmationEmail(Order $order)
    {

        $body = $this->templating->render('emails/print_confirmation.html.twig', [
            'order' => $order
        ]);

        $message = (new \Swift_Message())
            ->setSubject('L\'impression de votre commande')
            ->setFrom($this->mailFrom, $this->mailFromNom)
            ->setTo($order->getUser()->getEmail())
            ->setBody($body, 'text/html');
        $this->mailer->send($message);
    }

    public function sendPrinterRequestEmail(RefPrinter $printer, AccessibilityRegister $register)
    {

        $body = $this->templating->render('emails/printer_request.html.twig', [
            'register' => $register
        ]);

        $message = (new \Swift_Message())
            ->setSubject('Demande d\'impression de registre(s) d\'Accessibilité')
            ->setFrom($this->mailFrom, $this->mailFromNom)
            ->setTo($printer->getEmail())
            ->setBody($body, 'text/html');
        $this->mailer->send($message);
    }

    public function sendMydlCheckEmail(AccessibilityRegister $register, Exchange $exchange, User $user)
    {
        $body = $this->templating->render('emails/mydl_check.html.twig', [
            'register' => $register,
            'exchange' => $exchange,
        ]);

        $message = (new \Swift_Message())
            ->setSubject("Réponse à votre demande de vérification - Registre d'Accessibilité")
            ->setFrom($this->mailFrom, $this->mailFromNom)
            ->setTo($user->getEmail())
            ->setBody($body, 'text/html');
        $this->mailer->send($message);
    }

}